#!/bin/bash

# ===============================
# Settings
# ===============================
VERSION="Apple SHARP Auto Installer for Mac Ver0.3"
BASE="$HOME/SHARP"
CONDA_ROOT="$BASE/miniconda"
ENV_PATH="$CONDA_ROOT/envs/sharp"
PYTHON_VER="3.11"

echo "==============================="
echo "$VERSION"
echo "==============================="

# 1. License Agreement (User Prompt)
echo "This installer uses Miniconda3 setup."
echo "By continuing this installation you are accepting this license agreement:
-------------------------------------------------------------------------
MINICONDA END USER LICENSE AGREEMENT

Copyright Notice: Miniconda(R) (C) 2015, Anaconda, Inc.
All rights reserved. Miniconda(R) is licensed, not sold.
Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer;
2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution;
3. The name Anaconda, Inc. or Miniconda(R) may not be used to endorse or promote products derived from this software without specific prior written permission from Anaconda, Inc.; and
4. Miniconda(R) may not be used to access or allow third parties to access Anaconda package repositories if such use would circumvent paid licensing requirements or is otherwise restricted by the Anaconda Terms of Service.

DISCLAIMER: THIS SOFTWARE IS PROVIDED BY ANACONDA "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE , AND NON-INFRINGEMENT ARE DISCLAIMED. IN NO EVENT SHALL ANACONDA BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF MINICONDA(R), EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
-------------------------------------------------------------------------"
read -p "Do you agree to this LICENSE AGREEMENT? (y/N): " answer
if [[ ! "$answer" =~ ^[Yy]$ ]]; then
    echo "Installation cancelled."
    exit 1
fi

# Create base directory
mkdir -p "$BASE"
cd "$BASE"

# 2. Miniconda Installation
ARCH=$(uname -m)
if [ "$ARCH" == "arm64" ]; then
    MINICONDA_URL="https://repo.anaconda.com/miniconda/Miniconda3-latest-MacOSX-arm64.sh"
else
    MINICONDA_URL="https://repo.anaconda.com/miniconda/Miniconda3-latest-MacOSX-x86_64.sh"
fi

if [ ! -d "$CONDA_ROOT" ]; then
    echo "--- Downloading and Installing Miniconda ---"
    curl -L "$MINICONDA_URL" -o miniconda.sh
    bash miniconda.sh -b -u -p "$CONDA_ROOT"
    rm miniconda.sh
fi

CONDA_BIN="$CONDA_ROOT/bin/conda"

# 3-0. FINAL GUARD: Fix broken Conda TOS cache permission
TOS_CACHE="$HOME/Library/Caches/conda-anaconda-tos"
if [ -e "$TOS_CACHE" ]; then
    if [ ! -w "$TOS_CACHE" ]; then
        echo ""
        echo "============================================"
        echo "ERROR: Conda TOS cache permission is broken."
        echo "This is caused by running conda with sudo."
        echo ""
        echo "Please run the following command ONCE:"
        echo "  sudo rm -rf $TOS_CACHE"
        echo "Then re-run this installer."
        echo "============================================"
        exit 1
    fi
fi

# 3-1. Initialize conda (required before TOS)
"$CONDA_BIN" config --set always_yes yes >/dev/null
"$CONDA_BIN" info >/dev/null

# 3-2. Accept Anaconda Terms of Service
echo "--- Accepting Anaconda Terms of Service ---"
"$CONDA_BIN" tos accept --override-channels --channel https://repo.anaconda.com/pkgs/main
"$CONDA_BIN" tos accept --override-channels --channel https://repo.anaconda.com/pkgs/r

# 4. Create Environment (Using Path-based setup for stability)
echo "--- Creating Conda environment at $ENV_PATH ---"
rm -rf "$ENV_PATH"
"$CONDA_BIN" create -y -p "$ENV_PATH" python="$PYTHON_VER"

# 5. SHARP Download & Install
echo "--- Downloading SHARP Source ---"
curl -L "https://github.com/apple/ml-sharp/archive/refs/heads/main.zip" -o sharp.zip
unzip -o sharp.zip
SHARP_DIR_NAME=$(find . -maxdepth 1 -type d -name "ml-sharp*" | head -n 1)
rm sharp.zip

echo "--- Installing SHARP Requirements ---"
# Move to source dir and install
cd "$SHARP_DIR_NAME"
"$CONDA_BIN" run -p "$ENV_PATH" pip install --upgrade pip
"$CONDA_BIN" run -p "$ENV_PATH" pip install -e .

# 6. Create Launcher Script
cat << 'EOLAUNCH' > "$BASE/launcher.sh"
#!/bin/bash
set -e

INPUT_PATH="$1"

if [ -z "$INPUT_PATH" ]; then
    echo "Error: No folder dropped."
    exit 1
fi

SHARP_BIN="/Users/macminit2/SHARP/miniconda/envs/sharp/bin/sharp"

if [ ! -x "$SHARP_BIN" ]; then
    echo "Error: sharp command not found:"
    echo "  $SHARP_BIN"
    exit 1
fi

OUTPUT_PATH="$INPUT_PATH/sharp_output"
mkdir -p "$OUTPUT_PATH"

echo "--- SHARP Processing Start ---"
echo "Input:  $INPUT_PATH"
echo "Output: $OUTPUT_PATH"
echo "------------------------------"

"$SHARP_BIN" predict -i "$INPUT_PATH" -o "$OUTPUT_PATH"

echo ""
echo "Processing is complete!"
echo "Press any key to close."
read -n 1
EOLAUNCH

chmod +x "$BASE/launcher.sh"


# 7. Create Drag & Drop App on Desktop
APP_PATH="$HOME/Desktop/SHARP_DragDrop.app"
rm -rf "$APP_PATH"
osacompile -o "$APP_PATH" <<END
on open some_items
    set target_folder to POSIX path of (item 1 of some_items)
    set cmd to "bash \"$BASE/launcher.sh\" " & quoted form of target_folder & "; exit"
    tell application "Terminal"
        activate
        do script cmd
    end tell
end open
on run
    display dialog "Please drag and drop a folder containing images onto this icon." buttons {"OK"} default button "OK"
end run
END

echo ""
echo "===================================================="
echo "Installation completed successfully!"
echo "Please drag and drop a folder containing images"
echo "onto the 'SHARP_DragDrop' app on your Desktop."
echo "===================================================="